function stdcell_mv_crossvac_gui
% Majority-Voter "Cross" GUI + rotated copy + 4 layered corner fillers (4 layers)
% CROSSVAC variant: the central block has an HfO2 mid layer with a cross-shaped
% vacuum (horizontal + vertical trenches, each with width L_vac), instead of a
% full square vacuum. The rest is identical to the STANDARD MV GUI.
%
% Buttons:
%  - "Export SDE.txt": exports a parametric SDE deck with per-cell named contacts for any N.
%  - "Export Sdevice.txt": exports a parametric SDevice deck (physics + solve) for any N.
%
% Contact naming (per-branch, per index):
%   TopContact_south_1..N, TopContact_north_1..N, TopContact_west_1..N, TopContact_east_1..N, and TopContact_X (center).
%   Index: 1 = far end, N = near the center.

%% ====== DEFAULT PARAMETERS ======
P = struct( ...
    'L',24, 'W',10, 'H',2, ...                     % SiO2 base (X-length, Y-length, Z-thickness) in nm
    'origin',[0 0 0], ...                          % [x0 y0 z0] in nm
    'H_Ti_bottom',1, ...                           % bottom Titanium thickness (Z) in nm
    'H_gold_bottom',2, ...                         % bottom Gold thickness (Z) in nm
    'L_vac',4, 'H_vac',3, ...                      % trench width & thickness (used also for central cross)
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % top Gold cap (X-length, Y-length, Z-thickness) in nm
    'T_film',0.5, ...                              % Titanium rim film thickness (rim) in nm
    'CapSide_center',22 ...                        % square top cap side for CENTER block (nm)
);
Nmin = 3; Ndef = 3;   % cells per branch (minimum 3)

%% ====== Colors / style ======
[c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw] = colorsAndStyle();
F = faces6();

%% ====== GUI ======
ui = uifigure('Name','Majority-Voter (CrossVac center) + Rotated + Layered Fillers','Color','w','Position',[80 80 1320 760]);
gl = uigridlayout(ui,[2 2]); gl.RowHeight={'1x',56}; gl.ColumnWidth={520,'1x'};

% --- Controls panel
ctrl = uipanel(gl,'Title','Parameters','FontWeight','bold');
ctrl.Layout.Row=1; ctrl.Layout.Column=1;

cg = uigridlayout(ctrl,[23 2]);
cg.RowHeight = repmat({28},1,23);
cg.ColumnWidth = {260, 220};

r=1;
ef_L   = addField(cg, r, 'L  [nm]',         P.L,            [0 Inf]); r=r+1;
ef_W   = addField(cg, r, 'W  [nm]',         P.W,            [0 Inf]); r=r+1;
ef_H   = addField(cg, r, 'H  [nm]',         P.H,            [0 Inf]); r=r+1;

ef_Hti = addField(cg, r, 'H_{Ti,bot} [nm]', P.H_Ti_bottom,  [0 Inf]); r=r+1;
ef_Hau = addField(cg, r, 'H_{Au,bot} [nm]', P.H_gold_bottom,[0 Inf]); r=r+1;

ef_Lv  = addField(cg, r, 'L_{vac} [nm]',    P.L_vac,        [0 Inf]); r=r+1;
ef_Hv  = addField(cg, r, 'H_{vac} [nm]',    P.H_vac,        [0 Inf]); r=r+1;

ef_Lat = addField(cg, r, 'L_{Au,top} [nm]', P.L_Au_top,     [0 Inf]); r=r+1;
ef_Wat = addField(cg, r, 'W_{Au,top} [nm]', P.W_Au_top,     [0 Inf]); r=r+1;
ef_Hat = addField(cg, r, 'H_{Au,top} [nm]', P.H_Au_top,     [0 Inf]); r=r+1;

ef_Tf  = addField(cg, r, 'T_{film,Ti} [nm]',P.T_film,       [0 Inf]); r=r+1;

% Center block square cap side
ef_CapC = addField(cg, r, 'CapSide_{center} [nm]', P.CapSide_center, [0 Inf]); r=r+1;

% Origin
h = uilabel(cg,'Text','origin x0 [nm]','HorizontalAlignment','right','FontWeight','bold'); h.Layout.Row=r; h.Layout.Column=1;
ef_x0 = uieditfield(cg,'numeric','Value',P.origin(1),'Limits',[-Inf Inf]); ef_x0.Layout.Row=r; ef_x0.Layout.Column=2; r=r+1;

h = uilabel(cg,'Text','origin y0 [nm]','HorizontalAlignment','right','FontWeight','bold'); h.Layout.Row=r; h.Layout.Column=1;
ef_y0 = uieditfield(cg,'numeric','Value',P.origin(2),'Limits',[-Inf Inf]); ef_y0.Layout.Row=r; ef_y0.Layout.Column=2; r=r+1;

h = uilabel(cg,'Text','origin z0 [nm]','HorizontalAlignment','right','FontWeight','bold'); h.Layout.Row=r; h.Layout.Column=1;
ef_z0 = uieditfield(cg,'numeric','Value',P.origin(3),'Limits',[-Inf Inf]); ef_z0.Layout.Row=r; ef_z0.Layout.Column=2; r=r+1;

% Cells per branch
h = uilabel(cg,'Text','Cells/branch N (>=3)','HorizontalAlignment','right','FontWeight','bold'); h.Layout.Row=r; h.Layout.Column=1;
ef_N = uieditfield(cg,'numeric','Limits',[Nmin Inf],'RoundFractionalValues','on','Value',Ndef);
ef_N.Layout.Row=r; ef_N.Layout.Column=2; r=r+1;

% --- Buttons
btnPanel = uipanel(gl,'BorderType','none'); btnPanel.Layout.Row=2; btnPanel.Layout.Column=[1 2];
bgl = uigridlayout(btnPanel,[1 7]); bgl.ColumnWidth={'fit','fit','fit','fit','1x','fit','fit'};
uibutton(bgl,'Text','Reset default','ButtonPushedFcn',@(~,~)resetAll());
uibutton(bgl,'Text','Update view','BackgroundColor',[0.1 0.5 0.1],'FontColor','w','ButtonPushedFcn',@(~,~)updateAndDraw());
uibutton(bgl,'Text','Export SDE.txt','BackgroundColor',[0.10 0.35 0.80],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDE());
uibutton(bgl,'Text','Export Sdevice.txt','BackgroundColor',[0.80 0.35 0.10],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDEVICE());
uilabel(bgl,'Text',''); % spacer
uibutton(bgl,'Text','Save PNG','ButtonPushedFcn',@(~,~)savePNG());
uibutton(bgl,'Text','Close','ButtonPushedFcn',@(~,~)close(ui));

% --- Axes area
axPanel = uipanel(gl,'Title','Preview — Majority-Voter (CrossVac center) + Layered Fillers','FontWeight','bold');
axPanel.Layout.Row=1; axPanel.Layout.Column=2;
ax = uiaxes('Parent',axPanel,'BackgroundColor','w'); ax.Units='normalized'; ax.Position=[0 0 1 1];

% Auto-update on parameter change
watch = [ef_L,ef_W,ef_H,ef_Hti,ef_Hau,ef_Lv,ef_Hv,ef_Lat,ef_Wat,ef_Hat,ef_Tf,ef_CapC,ef_x0,ef_y0,ef_z0,ef_N];
for k=1:numel(watch), watch(k).ValueChangedFcn = @(~,~)updateAndDraw(); end

% Initial draw
updateAndDraw();

%% ===== Nested GUI helpers =====
    function ef = addField(parent, row, lbl, val, lim)
        % Create a numeric field with right-aligned label
        hLbl = uilabel(parent,'Text',lbl,'HorizontalAlignment','right','FontWeight','bold');
        hLbl.Layout.Row = row; hLbl.Layout.Column = 1;
        ef = uieditfield(parent,'numeric','Value',val,'Limits',lim, ...
            'LowerLimitInclusive','on','UpperLimitInclusive','on','ValueDisplayFormat','%.6g');
        ef.Layout.Row = row; ef.Layout.Column = 2;
    end

    function resetAll()
        % Reset all controls to defaults
        P = struct('L',24,'W',10,'H',2,'origin',[0 0 0], ...
                   'H_Ti_bottom',1,'H_gold_bottom',2, ...
                   'L_vac',4,'H_vac',3, ...
                   'L_Au_top',16,'W_Au_top',8,'H_Au_top',4, ...
                   'T_film',0.5, 'CapSide_center',22);
        ef_L.Value=P.L; ef_W.Value=P.W; ef_H.Value=P.H;
        ef_Hti.Value=P.H_Ti_bottom; ef_Hau.Value=P.H_gold_bottom;
        ef_Lv.Value=P.L_vac; ef_Hv.Value=P.H_vac;
        ef_Lat.Value=P.L_Au_top; ef_Wat.Value=P.W_Au_top; ef_Hat.Value=P.H_Au_top;
        ef_Tf.Value=P.T_film;
        ef_CapC.Value=P.CapSide_center;
        ef_x0.Value=P.origin(1); ef_y0.Value=P.origin(2); ef_z0.Value=P.origin(3);
        ef_N.Value = Ndef;
        updateAndDraw();
    end

    function savePNG()
        % Export current view as PNG
        [f,p] = uiputfile('mv_crossvac_filler_layered.png','Save as');
        if isequal(f,0), return; end
        exportgraphics(ax, fullfile(p,f), 'Resolution',300);
    end

    % ========================= SDE EXPORTER (named contacts, any N) =========================
    function exportSDE()
        % === Gather current parameters from controls ===
        P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
        P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
        P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
        P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
        P.T_film = ef_Tf.Value; P.CapSide_center = ef_CapC.Value;
        P.origin = [ef_x0.Value, ef_y0.Value, ef_z0.Value];
        N = max(Nmin, round(ef_N.Value));  % enforce minimum

        % === Convert nm -> µm ===
        um = @(x) x/1000;
        L_um   = um(P.L);        W_um   = um(P.W);        H_um   = um(P.H);
        HTi_um = um(P.H_Ti_bottom);
        HAuB_um= um(P.H_gold_bottom);
        Lvac_um= um(P.L_vac);    Hvac_um= um(P.H_vac);
        LAuT_um= um(P.L_Au_top); WAuT_um= um(P.W_Au_top); HAuT_um= um(P.H_Au_top);
        Tfilm_um = um(P.T_film); CapC_um = um(P.CapSide_center);
        x0_um = um(P.origin(1)); y0_um = um(P.origin(2)); z0_um = um(P.origin(3));

        % === file name like your pattern ===
        defName = sprintf('MV_CrossVac_N=%d Celle.txt', N);
        [f,p] = uiputfile(defName,'Export SDE.txt');
        if isequal(f,0), return; end
        fid = fopen(fullfile(p,f),'w');
        if fid<0
            uialert(ui,'Cannot write the file.','Error','Icon','error'); return;
        end

        fw = @(x) sprintf('%.6f',x); % 6 decimals

        try
            % ===== HEADER =====
            fprintf(fid,';; =====================================================================\n');
            fprintf(fid,';; SENTARUS SDE — Majority-Voter CrossVac (N=%d) + Per-cell Named Contacts\n', N);
            fprintf(fid,';; Units: micrometers (µm)\n');
            fprintf(fid,';; Top contacts defined:\n');
            fprintf(fid,';;   south_1..%d, north_1..%d, west_1..%d, east_1..%d, and center X.\n',N,N,N,N);
            fprintf(fid,';; Index: 1 = far end, %d = near the center.\n', N);
            fprintf(fid,';; =====================================================================\n\n');

            fprintf(fid,'(sde:clear)\n\n');

            % ===== MAIN PARAMETERS =====
            fprintf(fid,'; -------------------------------\n; MAIN PARAMETERS (µm)\n; -------------------------------\n');
            fprintf(fid,'(sde:define-parameter "BottomDielectricWidth"\t%s)\n', fw(L_um));
            fprintf(fid,'(sde:define-parameter "CellLength"\t\t\t%s)\n', fw(W_um));
            fprintf(fid,'(sde:define-parameter "BottomDielectricThickness"\t%s)\n', fw(H_um));
            fprintf(fid,'(sde:define-parameter "AdhesionLayerThickness"\t\t%s)\n', fw(HTi_um));
            fprintf(fid,'(sde:define-parameter "GoldLayerThickness"\t\t%s)\n', fw(HAuB_um));
            fprintf(fid,'(sde:define-parameter "TrenchMiddleWidth"\t\t%s)\n', fw(Lvac_um));
            fprintf(fid,'(sde:define-parameter "TrenchWallHeight"\t\t%s)\n', fw(Hvac_um));
            fprintf(fid,'(sde:define-parameter "TopGoldWidthX"\t\t\t%s)\n', fw(LAuT_um));
            fprintf(fid,'(sde:define-parameter "TopGoldHeightY"\t\t%s)\n', fw(WAuT_um));
            fprintf(fid,'(sde:define-parameter "TopGoldThicknessZ"\t\t%s)\n', fw(HAuT_um));
            fprintf(fid,'(sde:define-parameter "TitaniumRimThickness"\t\t%s)\n', fw(Tfilm_um));
            fprintf(fid,'(sde:define-parameter "CenterCapSide"\t\t\t%s)\n', fw(CapC_um));
            fprintf(fid,'(sde:define-parameter "x0"\t\t\t\t%s)\n', fw(x0_um));
            fprintf(fid,'(sde:define-parameter "y0"\t\t\t\t%s)\n', fw(y0_um));
            fprintf(fid,'(sde:define-parameter "z0"\t\t\t\t%s)\n', fw(z0_um));
            fprintf(fid,'(define Ncells %d)\n', N);
            fprintf(fid,'(sde:define-parameter "BlockMaterial"\t\t"HfO2")\n\n');

            % ===== DERIVED & CONSTANTS =====
            fprintf(fid,'; -------------------------------\n; DERIVED & CONSTANTS\n; -------------------------------\n');
            fprintf(fid,'(sde:define-parameter "TrenchWallWidth"\n  (/ (- BottomDielectricWidth TrenchMiddleWidth) 2.0))\n\n');
            fprintf(fid,'(define H_tot (+ BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness TrenchWallHeight TopGoldThicknessZ))\n');
            fprintf(fid,'(define RimPackWidth (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)))\n');
            fprintf(fid,'(define Pitch (if (> CellLength RimPackWidth)\n');
            fprintf(fid,'               (min CellLength (* 0.5 (+ CellLength TopGoldHeightY (* 2.0 TitaniumRimThickness))))\n');
            fprintf(fid,'               CellLength))\n\n');

            % ===== HELPERS =====
            fprintf(fid,'; -------------------------------\n; HELPERS\n; -------------------------------\n');
            fprintf(fid,'(define (mk-cuboid name mat x y z dx dy dz)\n');
            fprintf(fid,'  (sdegeo:create-cuboid (position x y z)\n');
            fprintf(fid,'                        (position (+ x dx) (+ y dy) (+ z dz))\n');
            fprintf(fid,'                        mat))\n\n');

            % ===== CONTACT SETS =====
            fprintf(fid,'; -------------------------------\n; CONTACT SETS\n; -------------------------------\n');
            fprintf(fid,'(sdegeo:define-contact-set "BottomContact" 4 (color:rgb 1 0 0) "##")\n\n');
            for side = ["south","north","west","east"]
                for k = 1:N
                    fprintf(fid,'(sdegeo:define-contact-set "TopContact_%s_%d" 4 (color:rgb 0 0 1) "##")\n', side, k);
                end
                fprintf(fid,'\n');
            end
            fprintf(fid,'(sdegeo:define-contact-set "TopContact_X" 4 (color:rgb 0 0 1) "##")\n\n');

            % ===== CELL +Y =====
            writeCellY(fid);

            % ===== CELL +X =====
            writeCellX(fid);

            % ===== CORNER FILLERS =====
            writeCornerLayers(fid);

            % ===== GEOMETRY BUILD (parametric N) =====
            fprintf(fid,';; =====================================================================\n;; GEOMETRY BUILD (N=%d)\n;; =====================================================================\n\n', N);

            % SOUTH branch (Y-): cut..cut..std near center
            fprintf(fid,';; SOUTH branch (Y-): cut..cut..std near center\n');
            fprintf(fid,'(define y_cursor y0)\n');
            for k = 1:N-1
                fprintf(fid,'(CellY "Ypre%d" x0 y_cursor z0 Pitch #t "TopContact_south_%d") (set! y_cursor (+ y_cursor Pitch))\n', k, k);
            end
            fprintf(fid,'(CellY "Ypre%d" x0 y_cursor z0 CellLength #f "TopContact_south_%d") (set! y_cursor (+ y_cursor CellLength))\n\n', N, N);

            % CENTER BLOCK (CROSSVAC)
            writeCenterBlock_CrossVac(fid);

            % NORTH branch (Y+): std near center, then cuts (labels descending)
            fprintf(fid,'; NORTH branch (Y+): std near center, then cuts\n');
            fprintf(fid,'(set! y_cursor (+ yB BottomDielectricWidth))\n');
            fprintf(fid,'(CellY "Ypost1" x0 y_cursor z0 CellLength #f "TopContact_north_%d") (set! y_cursor (+ y_cursor CellLength))\n', N);
            for k = N-1:-1:1
                fprintf(fid,'(CellY "Ypost%d" x0 y_cursor z0 Pitch #t "TopContact_north_%d") (set! y_cursor (+ y_cursor Pitch))\n', (N-k+1), k);
            end
            fprintf(fid,'\n');

            % EAST branch (X+): std then cuts
            fprintf(fid,';; EAST branch (X+): std near center, then cuts\n');
            fprintf(fid,'(define xB2 (+ xB BottomDielectricWidth))\n');
            fprintf(fid,'(define x_cursor xB2)\n');
            fprintf(fid,'(CellX "Xpos1" x_cursor yB z0 CellLength #f "TopContact_east_1") (set! x_cursor (+ x_cursor CellLength))\n');
            for k = 2:N
                fprintf(fid,'(CellX "Xpos%d" x_cursor yB z0 Pitch #t "TopContact_east_%d") (set! x_cursor (+ x_cursor Pitch))\n', k, k);
            end
            fprintf(fid,'\n');

            % WEST branch (X-): std then cuts to the left
            fprintf(fid,';; WEST branch (X-): std near center, then cuts to the left\n');
            fprintf(fid,'(set! x_cursor xB)\n');
            fprintf(fid,'(define x_origin (- x_cursor CellLength))\n');
            fprintf(fid,'(CellX "Xneg%d" x_origin yB z0 CellLength #f "TopContact_west_%d") (set! x_cursor x_origin)\n', N, N);
            for k = N-1:-1:1
                fprintf(fid,'(set! x_origin (- x_cursor Pitch))\n');
                fprintf(fid,'(CellX "Xneg%d" x_origin yB z0 Pitch #t "TopContact_west_%d") (set! x_cursor x_origin)\n', k, k);
            end
            fprintf(fid,'\n');

            % Corner fillers with parametric span
            fprintf(fid,'(define yB2 (+ yB BottomDielectricWidth))\n');
            fprintf(fid,'(define arm_span (+ CellLength (* (- Ncells 1) Pitch)))\n');
            fprintf(fid,'(CornerLayers "F_NE" xB2 yB2 z0 arm_span)\n');
            fprintf(fid,'(CornerLayers "F_NW" (- xB arm_span) yB2 z0 arm_span)\n');
            fprintf(fid,'(CornerLayers "F_SW" (- xB arm_span) (- yB arm_span) z0 arm_span)\n');
            fprintf(fid,'(CornerLayers "F_SE" xB2 (- yB arm_span) z0 arm_span)\n\n');

            % ===== MESH =====
            fprintf(fid,'; =====================================================================\n; MESH REFINEMENT\n; =====================================================================\n');
            fprintf(fid,'(sdedr:define-refinement-function "RFn" "MaxLenInt" "Gold" "Vacuum"   0.0001 "DoubleSide")\n');
            fprintf(fid,'(sdedr:define-refinement-function "RFn" "MaxLenInt" "Gold" "Titanium" 0.0001)\n\n');

            % ===== BUILD =====
            fprintf(fid,'; =====================================================================\n; BUILD\n; =====================================================================\n');
            fprintf(fid,'(sde:build-mesh "n@node@")\n');

            fclose(fid);
            uialert(ui,sprintf('SDE exported for N=%d (CrossVac center).',N),'Export SDE');
        catch ME
            fclose(fid);
            uialert(ui,['Error during SDE export: ' ME.message],'Error','Icon','error');
        end

        % ---- local writers (full blocks) ----
        function writeCellY(fid)
            fprintf(fid,'; -------------------------------\n; CELL +Y (with named top contact)\n; -------------------------------\n');
            fprintf(fid,'(define (CellY prefix x y z Wcell isCut cname)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_SiO2") "SiO2" x y z BottomDielectricWidth Wcell BottomDielectricThickness)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) BottomDielectricWidth Wcell AdhesionLayerThickness)\n');
            fprintf(fid,'  (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) BottomDielectricWidth Wcell GoldLayerThickness))\n');
            fprintf(fid,'  (sdegeo:set-current-contact-set "BottomContact")\n');
            fprintf(fid,'  (sdegeo:set-contact AUB "BottomContact")\n\n');

            fprintf(fid,'  (define ztop (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
            fprintf(fid,'  (define Lvac (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
            fprintf(fid,'  (define Lleft  (/ (- BottomDielectricWidth Lvac) 2.0))\n');
            fprintf(fid,'  (define Lright (- BottomDielectricWidth (+ Lleft Lvac)))\n');
            fprintf(fid,'  (if (> Lleft 0.0)  (mk-cuboid (string-append prefix "_HfO2_L") "HfO2" x y ztop Lleft Wcell TrenchWallHeight))\n');
            fprintf(fid,'  (if (> Lvac 0.0)   (mk-cuboid (string-append prefix "_Vac")    "Vacuum" (+ x Lleft) y ztop Lvac Wcell TrenchWallHeight))\n');
            fprintf(fid,'  (if (> Lright 0.0) (mk-cuboid (string-append prefix "_HfO2_R") "HfO2" (+ x Lleft Lvac) y ztop Lright Wcell TrenchWallHeight))\n\n');

            fprintf(fid,'  (define zcap (+ ztop TrenchWallHeight))\n');
            fprintf(fid,'  (define xcap (+ x (/ (- BottomDielectricWidth TopGoldWidthX) 2.0)))\n');
            fprintf(fid,'  (define ycap (if isCut (+ y (- Wcell (+ TopGoldHeightY TitaniumRimThickness))) (+ y (/ (- CellLength TopGoldHeightY) 2.0))))\n');
            fprintf(fid,'  (define AUT (mk-cuboid (string-append prefix "_AuT") "Gold" xcap ycap zcap TopGoldWidthX TopGoldHeightY TopGoldThicknessZ))\n');
            fprintf(fid,'  (sdegeo:set-current-contact-set cname)\n');
            fprintf(fid,'  (sdegeo:set-contact AUT cname)\n\n');

            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFxL") "Titanium" (- xcap TitaniumRimThickness) ycap zcap TitaniumRimThickness TopGoldHeightY TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFxR") "Titanium" (+ xcap TopGoldWidthX) ycap zcap TitaniumRimThickness TopGoldHeightY TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFyF") "Titanium" (- xcap TitaniumRimThickness) (- ycap TitaniumRimThickness) zcap (+ TopGoldWidthX (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFyB") "Titanium" (- xcap TitaniumRimThickness) (+ ycap TopGoldHeightY) zcap (+ TopGoldWidthX (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n\n');

            fprintf(fid,'  (define xin_min (- xcap TitaniumRimThickness))\n');
            fprintf(fid,'  (define xin_max (+ xcap TopGoldWidthX  TitaniumRimThickness))\n');
            fprintf(fid,'  (define yin_min (- ycap TitaniumRimThickness))\n');
            fprintf(fid,'  (define yin_max (+ ycap TopGoldHeightY TitaniumRimThickness))\n');
            fprintf(fid,'  (define xb_min x) (define xb_max (+ x BottomDielectricWidth))\n');
            fprintf(fid,'  (define yb_min y) (define yb_max (+ y Wcell))\n');
            fprintf(fid,'  (define Tleft  (max 0.0 (- xin_min xb_min)))\n');
            fprintf(fid,'  (define Tright (max 0.0 (- xb_max xin_max)))\n');
            fprintf(fid,'  (define Tfront (max 0.0 (- yin_min yb_min)))\n');
            fprintf(fid,'  (define Tback  (max 0.0 (- yb_max yin_max)))\n');
            fprintf(fid,'  (define dxcore (max 0.0 (- xin_max xin_min)))\n');
            fprintf(fid,'  (if (> Tleft 0.0)  (mk-cuboid (string-append prefix "_SiO2FrL") "SiO2" xb_min yb_min zcap Tleft Wcell TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (> Tright 0.0) (mk-cuboid (string-append prefix "_SiO2FrR") "SiO2" xin_max yb_min zcap Tright Wcell TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (and (> Tfront 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrF") "SiO2" xin_min yb_min zcap dxcore Tfront TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (and (> Tback 0.0)  (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrB") "SiO2" xin_min yin_max zcap dxcore Tback TopGoldThicknessZ))\n');
            fprintf(fid,')\n\n');
        end

        function writeCellX(fid)
            fprintf(fid,'; -------------------------------\n; CELL +X (with named top contact)\n; -------------------------------\n');
            fprintf(fid,'(define (CellX prefix x y z WcellX isCut cname)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_SiO2") "SiO2" x y z WcellX BottomDielectricWidth BottomDielectricThickness)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) WcellX BottomDielectricWidth AdhesionLayerThickness)\n');
            fprintf(fid,'  (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) WcellX BottomDielectricWidth GoldLayerThickness))\n');
            fprintf(fid,'  (sdegeo:set-current-contact-set "BottomContact")\n');
            fprintf(fid,'  (sdegeo:set-contact AUB "BottomContact")\n\n');

            fprintf(fid,'  (define ztop (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
            fprintf(fid,'  (define Ly_vac (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
            fprintf(fid,'  (define Ly_left  (/ (- BottomDielectricWidth Ly_vac) 2.0))\n');
            fprintf(fid,'  (define Ly_right (- BottomDielectricWidth (+ Ly_left Ly_vac)))\n');
            fprintf(fid,'  (if (> Ly_left 0.0)  (mk-cuboid (string-append prefix "_HfO2_L") "HfO2" x y ztop WcellX Ly_left TrenchWallHeight))\n');
            fprintf(fid,'  (if (> Ly_vac 0.0)   (mk-cuboid (string-append prefix "_Vac")    "Vacuum" x (+ y Ly_left) ztop WcellX Ly_vac TrenchWallHeight))\n');
            fprintf(fid,'  (if (> Ly_right 0.0) (mk-cuboid (string-append prefix "_HfO2_R") "HfO2" x (+ y Ly_left Ly_vac) ztop WcellX Ly_right TrenchWallHeight))\n\n');

            fprintf(fid,'  (define zcap (+ ztop TrenchWallHeight))\n');
            fprintf(fid,'  (define xcap (if isCut (+ x (- WcellX (+ TopGoldHeightY TitaniumRimThickness))) (+ x (/ (- CellLength TopGoldHeightY) 2.0))))\n');
            fprintf(fid,'  (define ycap (+ y (/ (- BottomDielectricWidth TopGoldWidthX) 2.0)))\n');
            fprintf(fid,'  (define AUT (mk-cuboid (string-append prefix "_AuT") "Gold" xcap ycap zcap TopGoldHeightY TopGoldWidthX TopGoldThicknessZ))\n');
            fprintf(fid,'  (sdegeo:set-current-contact-set cname)\n');
            fprintf(fid,'  (sdegeo:set-contact AUT cname)\n\n');

            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFxL") "Titanium" (- xcap TitaniumRimThickness) ycap zcap TitaniumRimThickness TopGoldWidthX TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFxR") "Titanium" (+ xcap TopGoldHeightY) ycap zcap TitaniumRimThickness TopGoldWidthX TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFyF") "Titanium" (- xcap TitaniumRimThickness) (- ycap TitaniumRimThickness) zcap (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_TiFyB") "Titanium" (- xcap TitaniumRimThickness) (+ ycap TopGoldWidthX) zcap (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n\n');

            fprintf(fid,'  (define xin_min (- xcap TitaniumRimThickness))\n');
            fprintf(fid,'  (define xin_max (+ xcap TopGoldHeightY TitaniumRimThickness))\n');
            fprintf(fid,'  (define yin_min (- ycap TitaniumRimThickness))\n');
            fprintf(fid,'  (define yin_max (+ ycap TopGoldWidthX  TitaniumRimThickness))\n');
            fprintf(fid,'  (define xb_min x) (define xb_max (+ x WcellX))\n');
            fprintf(fid,'  (define yb_min y) (define yb_max (+ y BottomDielectricWidth))\n');
            fprintf(fid,'  (define Tleft  (max 0.0 (- xin_min xb_min)))\n');
            fprintf(fid,'  (define Tright (max 0.0 (- xb_max xin_max)))\n');
            fprintf(fid,'  (define Tfront (max 0.0 (- yin_min yb_min)))\n');
            fprintf(fid,'  (define Tback  (max 0.0 (- yb_max yin_max)))\n');
            fprintf(fid,'  (define dxcore (max 0.0 (- xin_max xin_min)))\n');
            fprintf(fid,'  (if (> Tleft 0.0)  (mk-cuboid (string-append prefix "_SiO2FrL") "SiO2" xb_min yb_min zcap Tleft BottomDielectricWidth TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (> Tright 0.0) (mk-cuboid (string-append prefix "_SiO2FrR") "SiO2" xin_max yb_min zcap Tright BottomDielectricWidth TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (and (> Tfront 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrF") "SiO2" xin_min yb_min zcap dxcore Tfront TopGoldThicknessZ))\n');
            fprintf(fid,'  (if (and (> Tback 0.0)  (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrB") "SiO2" xin_min yin_max zcap dxcore Tback TopGoldThicknessZ))\n');
            fprintf(fid,')\n\n');
        end

        function writeCornerLayers(fid)
            fprintf(fid,'; -------------------------------\n; CORNER FILLERS (4 layers)\n; -------------------------------\n');
            fprintf(fid,'(define (CornerLayers prefix x y z span)\n');
            fprintf(fid,'  (define zc z)\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_L1_SiO2") "SiO2" x y zc span span (+ BottomDielectricThickness AdhesionLayerThickness))\n');
            fprintf(fid,'  (set! zc (+ zc (+ BottomDielectricThickness AdhesionLayerThickness)))\n');
            fprintf(fid,'  (define GOLD_FILL (mk-cuboid (string-append prefix "_L2_Au") "Gold" x y zc span span GoldLayerThickness))\n');
            fprintf(fid,'  (sdegeo:set-current-contact-set "BottomContact")\n');
            fprintf(fid,'  (sdegeo:set-contact GOLD_FILL "BottomContact")\n');
            fprintf(fid,'  (set! zc (+ zc GoldLayerThickness))\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_L3_HfO2") "HfO2" x y zc span span TrenchWallHeight)\n');
            fprintf(fid,'  (set! zc (+ zc TrenchWallHeight))\n');
            fprintf(fid,'  (mk-cuboid (string-append prefix "_L4_SiO2") "SiO2" x y zc span span TopGoldThicknessZ))\n\n');
        end

        function writeCenterBlock_CrossVac(fid)
            fprintf(fid,'; ================= CENTER BLOCK — HfO2 mid + Cross vacuum + TopContact_X\n');
            fprintf(fid,'(define xB x0)\n(define yB y_cursor)\n\n');
            fprintf(fid,'(mk-cuboid "BLOCK_SiO2" "SiO2" xB yB z0 BottomDielectricWidth BottomDielectricWidth BottomDielectricThickness)\n');
            fprintf(fid,'(mk-cuboid "BLOCK_TiB" "Titanium" xB yB (+ z0 BottomDielectricThickness) BottomDielectricWidth BottomDielectricWidth AdhesionLayerThickness)\n');
            fprintf(fid,'(define AUB_CENTER (mk-cuboid "BLOCK_AuB" "Gold" xB yB (+ z0 BottomDielectricThickness AdhesionLayerThickness) BottomDielectricWidth BottomDielectricWidth GoldLayerThickness))\n');
            fprintf(fid,'(sdegeo:set-current-contact-set "BottomContact")\n(sdegeo:set-contact AUB_CENTER "BottomContact")\n');
            fprintf(fid,'(define z_topBottom (+ z0 BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');

            % HfO2 mid slab
            fprintf(fid,'(mk-cuboid "BLOCK_mid" "HfO2" xB yB z_topBottom BottomDielectricWidth BottomDielectricWidth TrenchWallHeight)\n');

            % Cross vacuum (horizontal then vertical), each limited to the block size
            fprintf(fid,'(define Wy_h (min TrenchMiddleWidth BottomDielectricWidth))\n');
            fprintf(fid,'(define y_h (+ yB (/ (- BottomDielectricWidth Wy_h) 2.0)))\n');
            fprintf(fid,'(mk-cuboid "BLOCK_vacH" "Vacuum" xB y_h z_topBottom BottomDielectricWidth Wy_h TrenchWallHeight)\n');
            fprintf(fid,'(define Lx_v (min TrenchMiddleWidth BottomDielectricWidth))\n');
            fprintf(fid,'(define x_v (+ xB (/ (- BottomDielectricWidth Lx_v) 2.0)))\n');
            fprintf(fid,'(mk-cuboid "BLOCK_vacV" "Vacuum" x_v yB z_topBottom Lx_v BottomDielectricWidth TrenchWallHeight)\n');

            % Top cap + Ti rim + frame as usual
            fprintf(fid,'(define zcapC (+ z_topBottom TrenchWallHeight))\n');
            fprintf(fid,'(define capSide (min CenterCapSide BottomDielectricWidth))\n');
            fprintf(fid,'(define xcapC (+ xB (/ (- BottomDielectricWidth capSide) 2.0)))\n');
            fprintf(fid,'(define ycapC (+ yB (/ (- BottomDielectricWidth capSide) 2.0)))\n');
            fprintf(fid,'(define AuC (mk-cuboid "BLOCK_AuT" "Gold" xcapC ycapC zcapC capSide capSide TopGoldThicknessZ))\n');
            fprintf(fid,'(sdegeo:set-current-contact-set "TopContact_X")\n(sdegeo:set-contact AuC "TopContact_X")\n');
            fprintf(fid,'(mk-cuboid "BLOCK_TiL" "Titanium" (- xcapC TitaniumRimThickness) ycapC zcapC TitaniumRimThickness capSide TopGoldThicknessZ)\n');
            fprintf(fid,'(mk-cuboid "BLOCK_TiR" "Titanium" (+ xcapC capSide) ycapC zcapC TitaniumRimThickness capSide TopGoldThicknessZ)\n');
            fprintf(fid,'(mk-cuboid "BLOCK_TiF" "Titanium" (- xcapC TitaniumRimThickness) (- ycapC TitaniumRimThickness) zcapC (+ capSide (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
            fprintf(fid,'(mk-cuboid "BLOCK_TiB" "Titanium" (- xcapC TitaniumRimThickness) (+ ycapC capSide) zcapC (+ capSide (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');

            % Fill SiO2 frames around the rim if needed
            fprintf(fid,'(define xb_min xB) (define xb_max (+ xB BottomDielectricWidth))\n');
            fprintf(fid,'(define yb_min yB) (define yb_max (+ yB BottomDielectricWidth))\n');
            fprintf(fid,'(define xin_min (- xcapC TitaniumRimThickness))\n');
            fprintf(fid,'(define xin_max (+ xcapC capSide  TitaniumRimThickness))\n');
            fprintf(fid,'(define yin_min (- ycapC TitaniumRimThickness))\n');
            fprintf(fid,'(define yin_max (+ ycapC capSide  TitaniumRimThickness))\n');
            fprintf(fid,'(define TleftC  (max 0.0 (- xin_min xb_min)))\n');
            fprintf(fid,'(define TrightC (max 0.0 (- xb_max xin_max)))\n');
            fprintf(fid,'(define TfrontC (max 0.0 (- yin_min yb_min)))\n');
            fprintf(fid,'(define TbackC  (max 0.0 (- yb_max yin_max)))\n');
            fprintf(fid,'(define dxcoreC (max 0.0 (- xin_max xin_min)))\n');
            fprintf(fid,'(if (> TleftC 0.0)  (mk-cuboid "BLOCK_SiO2FrL" "SiO2" xb_min yb_min zcapC TleftC BottomDielectricWidth TopGoldThicknessZ))\n');
            fprintf(fid,'(if (> TrightC 0.0) (mk-cuboid "BLOCK_SiO2FrR" "SiO2" xin_max yb_min zcapC TrightC BottomDielectricWidth TopGoldThicknessZ))\n');
            fprintf(fid,'(if (and (> TfrontC 0.0) (> dxcoreC 0.0)) (mk-cuboid "BLOCK_SiO2FrF" "SiO2" xin_min yb_min zcapC dxcoreC TfrontC TopGoldThicknessZ))\n');
            fprintf(fid,'(if (and (> TbackC 0.0)  (> dxcoreC 0.0)) (mk-cuboid "BLOCK_SiO2FrB" "SiO2" xin_min yin_max zcapC dxcoreC TbackC TopGoldThicknessZ))\n\n');
        end
    end

    % ========================= SDEVICE EXPORTER (physics + solve, any N) =========================
    function exportSDEVICE()
        % Read N (enforce minimum)
        N = max(Nmin, round(ef_N.Value));

        defName = sprintf('MV_crossvac_sdevice_N=%d.txt', N);
        [f,p] = uiputfile(defName,'Export Sdevice.txt');
        if isequal(f,0), return; end
        fid = fopen(fullfile(p,f),'w');
        if fid<0
            uialert(ui,'Cannot write the file.','Error','Icon','error'); return;
        end

        % Cyclic voltages: S/H/R = +3/0/-3
        S = 3.0; H = 0.0; R = -3.0;
        phases = {'S','H','R'};
        vmap.S = S; vmap.H = H; vmap.R = R;

        phaseAt = @(i,startPhase) phases{ 1 + mod( (find(strcmp(phases,startPhase))-1) + (i-1), 3) };
        voltAt  = @(i,startPhase) vmap.( phaseAt(i,startPhase) );

        try
            % ===== File block =====
            fprintf(fid,'File {\n');
            fprintf(fid,'  Grid  \t= "n1_msh.tdr"\n');
            fprintf(fid,'  Plot  \t= "n@node@_clock_des.tdr"\n');
            fprintf(fid,'  Current   = "n@node@_clock_des.plt"\n');
            fprintf(fid,'  Parameter = "sdevice.par"\n');
            fprintf(fid,'}\n\n\n');

            % ===== Electrode block (all contacts) =====
            fprintf(fid,'Electrode {\n\n');
            fprintf(fid,'*BOTTOMCONTACT\n\n');
            fprintf(fid,'  { Name = "BottomContact"                \tVoltage = 0.0 }\n\n');
            fprintf(fid,'*INPUTS SEQUENCES TOPCONTACT\n\n');

            for side = ["south","north","west"]
                for k=1:N
                    fprintf(fid,'  { Name = "TopContact_%s_%d"   Voltage = 0.0 }\n', side, k);
                end
                fprintf(fid,'\n');
            end

            fprintf(fid,'*CENTER CELL TOPCONTACT\n\n');
            fprintf(fid,'  { Name = "TopContact_X"\tVoltage = 0.0 }\n\n\n');

            fprintf(fid,'*OUTPUT TOPCONTACT\n\n');
            for k=1:N
                fprintf(fid,'  { Name = "TopContact_east_%d"   Voltage = 0.0 }\n', k);
            end
            fprintf(fid,'\n}\n\n\n');

            % ===== Physics blocks =====
            fprintf(fid,'# ----------------------------\n# PHYSICS\n# ----------------------------\n');
            fprintf(fid,'Physics (Material="HfO2") {\n  CondInsulator\n}\n\n');
            fprintf(fid,'Physics (Material="SiO2") {\n  CondInsulator\n}\n\n');

            % ===== Thermode =====
            fprintf(fid,'Thermode {\n\n');
            for side = ["south","north","west"]
                for k=1:N
                    fprintf(fid,'  { Name = "TopContact_%s_%d"  Temperature = 300 SurfaceResistance = 1e-5 }\n', side, k);
                end
                fprintf(fid,'\n');
            end
            fprintf(fid,'  { Name = "TopContact_X"               \tTemperature = 300 SurfaceResistance = 1e-5 }\n\n');
            for k=1:N
                fprintf(fid,'  { Name = "TopContact_east_%d"  Temperature = 300 SurfaceResistance = 1e-5 }\n', k);
            end
            fprintf(fid,'\n  { Name = "BottomContact"              \tTemperature = 300 SurfaceResistance = 1e-5 }\n');
            fprintf(fid,'}\n\n');

            % ===== Plot block =====
            fprintf(fid,'# ----------------------------\n# PLOT\n# ----------------------------\n');
            fprintf(fid,'Plot {\n');
            fprintf(fid,'  Potential\n');
            fprintf(fid,'  ElectricField\n');
            fprintf(fid,'  DielectricConstant\n');
            fprintf(fid,'  Temperature\n');
            fprintf(fid,'  ConductionCurrentDensity/Vector\n');
            fprintf(fid,'  DisplacementCurrentDensity/Vector\n');
            fprintf(fid,'  TotalCurrent/Vector\n');
            fprintf(fid,'  SpaceCharge\n');
            fprintf(fid,'  Potential Doping\n');
            fprintf(fid,'  BandGap ElectronAffinity\n');
            fprintf(fid,'  ConductionBandEnergy ValenceBandEnergy\n');
            fprintf(fid,'}\n\n');

            % ===== Math =====
            fprintf(fid,'Math {\n  RelErrControl\n  Extrapolate\n}\n\n');

            % ===== Solve header (equilibrium) =====
            fprintf(fid,'Solve {\n');
            fprintf(fid,'  Coupled (Iterations= 100 LineSearchDamping= 1e-8) {Poisson}\n');
            fprintf(fid,'  Coupled{ Poisson Temperature Contact CondInsulator }\n');
            fprintf(fid,'  Plot(FilePrefix="n@node@_equilibrium")\n\n');

            %% ---------------- CONFIG 1 ----------------
            writeConfig(fid, N, 'S', voltAt, "cfg1");

            %% Return to 0 V
            writeAllZero(fid, N);

            %% ---------------- CONFIG 2 ----------------
            writeConfig(fid, N, 'R', voltAt, "cfg2");

            %% Return to 0 V
            writeAllZero(fid, N);

            %% ---------------- CONFIG 3 ----------------
            writeConfig(fid, N, 'H', voltAt, "cfg3");

            fprintf(fid,'}\n\n'); % end Solve

            fclose(fid);
            uialert(ui,sprintf('Sdevice deck exported for N=%d.',N),'Export Sdevice');
        catch ME
            fclose(fid);
            uialert(ui,['Error during Sdevice export: ' ME.message],'Error','Icon','error');
        end

        % ---- writers implementing the single running index logic ----
        function writeConfig(fid, N, startPhase, voltAtFcn, tag)
            fprintf(fid,'  # =====================================================================\n');
            fprintf(fid,'  # ==== CONFIG (%s) with startPhase=%s; inputs 1..N, center N+1, east N+2..N+1+N ====\n', char(tag), startPhase);
            fprintf(fid,'  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');
            fprintf(fid,'  Goal{ name = "BottomContact"                \tvoltage = 0.0 }\n\n');

            % Inputs (South, North, West): indices 1..N
            for side = ["south","north","west"]
                for idx = 1:N
                    v = voltAtFcn(idx, startPhase);
                    fprintf(fid,'  Goal{ name = "TopContact_%s_%d"\tvoltage = %.1f }\n', side, idx, v);
                end
                fprintf(fid,'\n');
            end

            % Center contact: index N+1
            vX = voltAtFcn(N+1, startPhase);
            fprintf(fid,'  Goal{ name = "TopContact_X"                 \tvoltage = %.1f }\n\n', vX);

            % East outputs: indices N+2 .. N+1+N mapped to east_1..east_N
            for k = 1:N
                vE = voltAtFcn(N+1+k, startPhase);
                fprintf(fid,'  Goal{ name = "TopContact_east_%d"\tvoltage = %.1f }\n', k, vE);
            end
            fprintf(fid,'\n\tplot { range=(0, 1) intervals= 1}\n');
            fprintf(fid,'  ){coupled { Poisson Temperature CondInsulator }}\n\n');
            fprintf(fid,'  Plot(FilePrefix="n@node@_%s")\n\n', char(tag));
        end

        function writeAllZero(fid, N)
            fprintf(fid,'  # ---- return to 0 V on all top contacts\n');
            fprintf(fid,'  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n');
            fprintf(fid,'  Goal{ name = "BottomContact"                \tvoltage = 0.0 }\n\n');

            for side = ["south","north","west"]
                for idx=1:N
                    fprintf(fid,'  Goal{ name = "TopContact_%s_%d"\tvoltage = 0.0 }\n', side, idx);
                end
                fprintf(fid,'\n');
            end
            fprintf(fid,'  Goal{ name = "TopContact_X"                 \tvoltage = 0.0 }\n\n');
            for idx=1:N
                fprintf(fid,'  Goal{ name = "TopContact_east_%d"\tvoltage = 0.0 }\n', idx);
            end
            fprintf(fid,'\n\tplot { range=(0, 1) intervals= 1}\n');
            fprintf(fid,'  ){coupled { Poisson Temperature CondInsulator }}\n\n');
        end
    end

    function updateAndDraw()
        % Read parameters and render the 3D preview
        P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
        P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
        P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
        P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
        P.T_film = ef_Tf.Value; P.CapSide_center = ef_CapC.Value;
        P.origin = [ef_x0.Value, ef_y0.Value, ef_z0.Value];
        N = max(Nmin, round(ef_N.Value)); ef_N.Value = N;

        % Reset axes
        if ~isgraphics(ax), ax = gca; end
        cla(ax,'reset'); delete(findall(ax,'Type','legend')); hold(ax,'on');

        % Geometry total height
        H_tot  = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

        % Central block (square L x L x H_tot)
        L_block = P.L; W_block = P.L;

        x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);

        % Cells BEFORE the block along +Y: i = 1..N (STANDARD)
        y_cursor = y0;
        G_pre = cell(1,N);
        for i=1:N
            G_pre{i} = buildCellVerts(P, [x0, y_cursor, z0]); % STANDARD
            y_cursor = y_cursor + P.W;
        end

        % Central block as CROSSVAC
        oB = [x0, y_cursor, z0];
        Gc = buildCentralCrossVacCellVerts(P, oB);
        y_after_block = oB(2) + W_block;

        % Cells AFTER the block along +Y: j = 1..N (STANDARD)
        G_post = cell(1,N);
        y_cursor = y_after_block;
        for j=1:N
            G_post{j} = buildCellVerts(P, [x0, y_cursor, z0]); % STANDARD
            y_cursor = y_cursor + P.W;
        end

        % Rotation center for X branches
        Crot = [oB(1) + L_block/2, oB(2) + W_block/2];

        % Rotate +90° around Z for the X branches
        Rz = [0 -1 0; 1 0 0; 0 0 1];
        G_pre_r  = cellfun(@(G) rotZ_around(G,Rz,Crot), G_pre, 'UniformOutput',false);
        G_post_r = cellfun(@(G) rotZ_around(G,Rz,Crot), G_post,'UniformOutput',false);

        % Corner fillers (4 layers) as opaque blocks
        arm_span = N * P.W;  % branch span from block edge (STANDARD)
        xB = oB(1); yB = oB(2); xB2 = xB + L_block; yB2 = yB + W_block;

        h1 = P.H + P.H_Ti_bottom;
        h2 = P.H_gold_bottom;
        h3 = P.H_vac;
        h4 = P.H_Au_top;

        o_NE = [xB2,            yB2,            z0];
        o_NW = [xB - arm_span,  yB2,            z0];
        o_SW = [xB - arm_span,  yB - arm_span,  z0];
        o_SE = [xB2,            yB - arm_span,  z0];

        FNE = buildFillerLayers(o_NE, arm_span, arm_span, [h1 h2 h3 h4]);
        FNW = buildFillerLayers(o_NW, arm_span, arm_span, [h1 h2 h3 h4]);
        FSW = buildFillerLayers(o_SW, arm_span, arm_span, [h1 h2 h3 h4]);
        FSE = buildFillerLayers(o_SE, arm_span, arm_span, [h1 h2 h3 h4]);

        % ==== DRAW ====
        drawGeom(ax,G_pre{1}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, true );  % legend once
        for i=2:N, drawGeom(ax,G_pre{i}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

        drawGeom_Center(ax,Gc, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F);

        for j=1:N, drawGeom(ax,G_post{j}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

        for i=1:N, drawGeom(ax,G_pre_r{i},  c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end
        for j=1:N, drawGeom(ax,G_post_r{j}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

        drawFillerLayers(ax, FNE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, 'FILLER (4 layers)');
        drawFillerLayers(ax, FNW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
        drawFillerLayers(ax, FSW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
        drawFillerLayers(ax, FSE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);

        axis(ax,'equal'); grid(ax,'on'); view(ax,35,25);
        xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
        title(ax, sprintf('Majority-Voter CrossVac — N=%d — H_{tot}=%.3g nm — Cap_{center}=%.3g nm', ...
                          N, H_tot, P.CapSide_center));

        branch_span = N * P.W;
        total_span  = 2*branch_span + W_block;
        marg   = 0.12 * [total_span, total_span, H_tot];
        xlim(ax,[P.origin(1) - total_span/2 - marg(1), P.origin(1) + total_span/2 + marg(1)]);
        ylim(ax,[P.origin(2) - marg(2),                P.origin(2) + total_span + marg(2)]);
        zlim(ax,[P.origin(3) - marg(3),                P.origin(3) + H_tot + marg(3)]);

        plot3(ax,P.origin(1),P.origin(2),P.origin(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
        legend(ax,'Location','bestoutside'); hold(ax,'off'); drawnow limitrate;
    end
end

% ===================== SUBFUNCTIONS =============================
function G = buildCellVerts(P, o)
% Standard rectangular cell (no CUT); returns a struct of vertex lists
V_base = vbox(o, P.L, P.W, P.H);
o_Ti   = o + [0 0 P.H];
V_Ti   = vbox(o_Ti, P.L, P.W, P.H_Ti_bottom);
o_Au   = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au   = vbox(o_Au, P.L, P.W, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
L_vac_eff = min(max(P.L_vac,0), P.L);
L_left  = max(0, (P.L - L_vac_eff)/2);
L_right = max(0, P.L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vbox(o_left,  L_left,    P.W, P.H_vac);
V_vac   = vbox(o_vac,   L_vac_eff, P.W, P.H_vac);
V_right = vbox(o_right, L_right,   P.W, P.H_vac);

x0_cap = o(1) + (P.L - P.L_Au_top)/2;
y0_cap = o(2) + (P.W - P.W_Au_top)/2;
z0_cap = z0_top + P.H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vbox(o_cap, P.L_Au_top, P.W_Au_top, P.H_Au_top);

o_film_xL = [x0_cap - P.T_film,      y0_cap,              z0_cap];
o_film_xR = [x0_cap + P.L_Au_top,    y0_cap,              z0_cap];
V_film_xL = vbox(o_film_xL, P.T_film,             P.W_Au_top, P.H_Au_top);
V_film_xR = vbox(o_film_xR, P.T_film,             P.W_Au_top, P.H_Au_top);
o_film_yF = [x0_cap - P.T_film,      y0_cap - P.T_film,   z0_cap];
o_film_yB = [x0_cap - P.T_film,      y0_cap + P.W_Au_top, z0_cap];
V_film_yF = vbox(o_film_yF, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);
V_film_yB = vbox(o_film_yB, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);

x_in_min = x0_cap - P.T_film; x_in_max = x0_cap + P.L_Au_top + P.T_film;
y_in_min = y0_cap - P.T_film; y_in_max = y0_cap + P.W_Au_top + P.T_film;

x_base_min = o(1); x_base_max = o(1) + P.L;
y_base_min = o(2); y_base_max = o(2) + P.W;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0,  y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  P.W, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, P.W, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

G.base  = V_base;  G.ti   = V_Ti;    G.au   = V_Au;
G.left  = V_left;  G.vac  = V_vac;   G.right= V_right;
G.cap   = V_cap;   G.fxL  = V_film_xL; G.fxR = V_film_xR;
G.fyF   = V_film_yF; G.fyB = V_film_yB;
G.frmL  = V_frame_xL; G.frmR = V_frame_xR;
G.frmF  = V_frame_yF; G.frmB = V_frame_yB;
end

function Gc = buildCentralCrossVacCellVerts(P, o)
% Central block variant: HfO2 mid slab + cross-shaped vacuum channels
Lside = P.L; Wside = P.L;

V_base = vbox(o, Lside, Wside, P.H);
o_Ti   = o + [0 0 P.H];
V_Ti   = vbox(o_Ti, Lside, Wside, P.H_Ti_bottom);
o_Au   = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au   = vbox(o_Au, Lside, Wside, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

% HfO2 full mid layer
V_mid  = vbox([o(1) o(2) z0_top], Lside, Wside, P.H_vac);

% Cross vacuum cutouts (width = L_vac, thickness = H_vac)
Lx_v = min(P.L_vac, Lside);
x_v  = o(1) + (Lside - Lx_v)/2;
V_vacV = vbox([x_v o(2) z0_top], Lx_v, Wside, P.H_vac);

Wy_h = min(P.L_vac, Wside);
y_h  = o(2) + (Wside - Wy_h)/2;
V_vacH = vbox([o(1) y_h z0_top], Lside, Wy_h, P.H_vac);

% Top cap (square) + Ti rim
capSide = min(P.CapSide_center, Lside);
x0_cap  = o(1) + (Lside - capSide)/2;
y0_cap  = o(2) + (Wside - capSide)/2;
z0_cap  = z0_top + P.H_vac;
V_cap   = vbox([x0_cap y0_cap z0_cap], capSide, capSide, P.H_Au_top);

V_film_xL = vbox([x0_cap - P.T_film, y0_cap,               z0_cap], P.T_film,             capSide, P.H_Au_top);
V_film_xR = vbox([x0_cap + capSide,  y0_cap,               z0_cap], P.T_film,             capSide, P.H_Au_top);
V_film_yF = vbox([x0_cap - P.T_film, y0_cap - P.T_film,    z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);
V_film_yB = vbox([x0_cap - P.T_film, y0_cap + capSide,     z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);

x_in_min = x0_cap - P.T_film; x_in_max = x0_cap + capSide + P.T_film;
y_in_min = y0_cap - P.T_film; y_in_max = y0_cap + capSide + P.T_film;
x_base_min = o(1); x_base_max = o(1) + Lside;
y_base_min = o(2); y_base_max = o(2) + Wside;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0, y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;
V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  Wside, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, Wside, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

Gc.base  = V_base;  Gc.ti   = V_Ti;    Gc.au   = V_Au;
Gc.mid   = V_mid;   Gc.vacH = V_vacH;  Gc.vacV = V_vacV;
Gc.cap   = V_cap;   Gc.fxL  = V_film_xL; Gc.fxR = V_film_xR;
Gc.fyF   = V_film_yF; Gc.fyB= V_film_yB;
Gc.frmL  = V_frame_xL; Gc.frmR = V_frame_xR;
Gc.frmF  = V_frame_yF; Gc.frmB = V_frame_yB;
end

function Gt = rotZ_around(G, Rz, Cxy)
% Rotate all vertex groups around Z by matrix Rz with respect to center Cxy (x,y)
fn = fieldnames(G);
for i=1:numel(fn)
    V = G.(fn{i});
    if isempty(V), Gt.(fn{i}) = []; continue; end
    Vc = V; Vc(:,1)=V(:,1)-Cxy(1); Vc(:,2)=V(:,2)-Cxy(2);
    Vrot = (Rz * Vc.').'; Vrot(:,1)=Vrot(:,1)+Cxy(1); Vrot(:,2)=Vrot(:,2)+Cxy(2);
    Gt.(fn{i}) = Vrot;
end
end

function drawGeom(ax,G, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, addLegend)
% Draw all parts of one linear cell geometry
drawPatch(ax,G.base,  c_base, edgeCol,lw, addLegend, 'SiO2',  1,      F);
drawPatch(ax,G.ti,    c_ti,   edgeCol,lw, addLegend, 'Ti',    1,      F);
drawPatch(ax,G.au,    c_au,   edgeCol,lw, addLegend, 'Au',    1,      F);
drawPatch(ax,G.left,  c_hfo2, edgeCol,lw, addLegend, 'HfO2',  1,      F);
drawPatch(ax,G.vac,   c_vac,  edgeCol,lw, addLegend, 'vacuum',a_vac,  F);
drawPatch(ax,G.right, c_hfo2, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.cap,   c_au,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fxL,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fxR,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fyF,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fyB,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmL,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmR,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmF,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmB,  c_base, edgeCol,lw, false,     '',      1,      F, true);
end

function drawGeom_Center(ax,Gc, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F)
% Draw central cross-vac block (with legend entries appearing once)
drawPatch(ax,Gc.base, c_base, edgeCol,lw, true,'SiO2', 1, F);
drawPatch(ax,Gc.ti,   c_ti,   edgeCol,lw, true,'Ti',   1, F);
drawPatch(ax,Gc.au,   c_au,   edgeCol,lw, true,'Au',   1, F);

drawPatch(ax,Gc.mid,  c_hfo2, edgeCol,lw, true,'HfO2', 1, F);
drawPatch(ax,Gc.vacH, c_vac,  edgeCol,lw, true,'vacuum',a_vac, F);
drawPatch(ax,Gc.vacV, c_vac,  edgeCol,lw, false,'', a_vac, F);

drawPatch(ax,Gc.cap, c_au, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.fxL, c_ti, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.fxR, c_ti, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.fyF, c_ti, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.fyB, c_ti, edgeCol,lw, false,'', 1, F);

drawPatch(ax,Gc.frmL, c_base, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.frmR, c_base, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.frmF, c_base, edgeCol,lw, false,'', 1, F);
drawPatch(ax,Gc.frmB, c_base, edgeCol,lw, false,'', 1, F);
end

function FL = buildFillerLayers(o, Lx, Wy, hvec)
% Build vertices for a 4-layer square filler of side 'span' starting at origin 'o'
z = o(3);
FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function V = vbox(o,Lx,Wy,Hz)
% Create vertex list (8 vertices) for an axis-aligned cuboid
V = [ o;
      o + [Lx 0 0];
      o + [0 Wy 0];
      o + [0 0 Hz];
      o + [Lx Wy 0];
      o + [Lx 0 Hz];
      o + [0 Wy Hz];
      o + [Lx Wy Hz] ];
end

function F = faces6()
% Quad faces for the 8-vertex cuboid
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F, offLegend)
% Draw a patch, optionally adding it to legend
if nargin<10, offLegend=false; end
if isempty(V), ph = []; return; end
if addToLegend && ~offLegend && ~isempty(name)
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
else
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
% Draw a 4-layer filler (all opaque). Only layer 1 can appear in the legend.
if nargin < 10, legendName = ''; end
if nargin < 11, offLegend = false; end

% Layer 1 (legend if requested)
if ~isempty(legendName) && ~offLegend
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1, ...
        'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',legendName);
else
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1, ...
        'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end

% Layers 2–4 (no legend)
patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end

function [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw] = colorsAndStyle()
c_base   = [0.55 0.35 0.20];   % SiO2
c_ti     = [0.20 0.20 0.20];   % Ti
c_au     = [1.00 0.84 0.00];   % Au
c_vac    = [0.00 0.60 0.00];   % Vacuum
a_vac    = 0.35;               % alpha for Vacuum
c_hfo2   = [0.80 0.65 0.45];   % HfO2
edgeCol = 'k'; lw = 1.0;
end
